class VerAInfacher {
  static MAX_TEXT_LEN = 5000;

  constructor() {
    this.tooltip = null;
    this.selectedText = '';
    this.init();
  }

  init() {
    // Event Listener für Textauswahl
    document.addEventListener('mouseup', this.handleTextSelection.bind(this));
    document.addEventListener('keyup', this.handleTextSelection.bind(this));
    
    // Event Listener um Tooltip zu verstecken bei Klicks außerhalb
    document.addEventListener('mousedown', this.handleMouseDown.bind(this));
    
    // Event Listener für Scroll und Resize um Tooltip zu verstecken
    window.addEventListener('scroll', this.hideTooltip.bind(this));
    window.addEventListener('resize', this.hideTooltip.bind(this));
  }

  handleTextSelection(event) {
    // Kurze Verzögerung, damit die Auswahl vollständig ist
    setTimeout(() => {
      const selection = window.getSelection();
      const selectedText = selection.toString().trim();

      if (selectedText.length > 0) {
        this.selectedText = selectedText;
        const explain = selectedText.length <= VerAInfacher.MAX_TEXT_LEN;
        this.showTooltip(selection, explain);
      } else {
        this.hideTooltip();
      }
    }, 10);
  }

  handleMouseDown(event) {
    // Tooltip nicht verstecken wenn auf den Button geklickt wird
    if (event.target.closest('.verainfacher-tooltip')) {
      return;
    }
    
    // Kurze Verzögerung, damit Textauswahl zuerst verarbeitet wird
    setTimeout(() => {
      const selection = window.getSelection();
      if (!selection.toString().trim()) {
        this.hideTooltip();
      }
    }, 10);
  }

  showTooltip(selection, explain = true) {
    this.hideTooltip(); // Entferne vorherigen Tooltip

    if (selection.rangeCount === 0) return;

    const range = selection.getRangeAt(0);
    const rect = range.getBoundingClientRect();

    // Erstelle Tooltip Element
    this.tooltip = document.createElement('div');
    this.tooltip.className = 'verainfacher-tooltip';

    const button = document.createElement('button');
    button.className = 'verainfacher-button';

    if (explain) {
      button.innerHTML = '🤖 Erkläre mir das...';
      button.style.backgroundColor = '#007bff'; // blau
      button.style.color = 'white';
      button.addEventListener('click', this.handleExplainClick.bind(this));
    } else {
      button.innerHTML = 'Leider ist der Text zu lang für eine Erklärung.';
      button.style.backgroundColor = '#dc3545'; // rot
      button.style.color = 'white';
      // Kein Klick-Handler
    }

    this.tooltip.appendChild(button);
    document.body.appendChild(this.tooltip);

    // Positioniere Tooltip
    this.positionTooltip(rect);
  }

  positionTooltip(selectionRect) {
    const tooltip = this.tooltip;
    const tooltipRect = tooltip.getBoundingClientRect();
    
    // Standard Position: über der Auswahl
    let top = selectionRect.top + window.scrollY - tooltipRect.height - 10;
    let left = selectionRect.left + window.scrollX + (selectionRect.width / 2) - (tooltipRect.width / 2);

    // Prüfe ob Tooltip über den oberen Rand hinausragt
    if (top < window.scrollY + 10) {
      // Zeige unter der Auswahl
      top = selectionRect.bottom + window.scrollY + 10;
    }

    // Prüfe horizontale Grenzen
    if (left < 10) {
      left = 10;
    } else if (left + tooltipRect.width > window.innerWidth - 10) {
      left = window.innerWidth - tooltipRect.width - 10;
    }

    tooltip.style.top = `${top}px`;
    tooltip.style.left = `${left}px`;
  }

  handleExplainClick(event) {
    event.preventDefault();
    event.stopPropagation();

    if (!this.selectedText) return;

    // Erstelle URL mit encoded Text
    const prompt = `Erkläre mir das: ${this.selectedText}`;
    const encodedPrompt = encodeURIComponent(prompt);
    const url = `https://www.verainfacher.de?prompt=${encodedPrompt}`;

    // Öffne neuen Tab und wechsle dorthin
    chrome.runtime.sendMessage({
      action: 'openTab',
      url: url
    });

    this.hideTooltip();
  }

  hideTooltip() {
    if (this.tooltip) {
      this.tooltip.remove();
      this.tooltip = null;
    }
  }
}

// Initialisiere die Extension
new VerAInfacher();
